<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\ProfessionalController;
use App\Http\Controllers\Api\ClientController;
use App\Http\Controllers\Api\ConsultationController;
use App\Http\Controllers\Api\PaymentController;
use App\Http\Controllers\Api\ReviewController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// Public API routes
Route::prefix('v1')->group(function () {
    // Authentication routes
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/logout', [AuthController::class, 'logout'])->middleware('auth:sanctum');
    Route::get('/user', [AuthController::class, 'user'])->middleware('auth:sanctum');
    
    // Password reset routes
    Route::post('/password/reset-request', [AuthController::class, 'sendResetLinkEmail']);
    Route::post('/password/reset', [AuthController::class, 'reset']);
    
    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        // Professional routes
        Route::prefix('professional')->name('api.professional.')->group(function () {
            Route::get('/dashboard', [ProfessionalController::class, 'dashboard']);
            Route::get('/profile', [ProfessionalController::class, 'profile']);
            Route::put('/profile', [ProfessionalController::class, 'updateProfile']);
            Route::get('/consultations', [ProfessionalController::class, 'consultations']);
            Route::get('/earnings', [ProfessionalController::class, 'earnings']);
            Route::get('/subscription', [ProfessionalController::class, 'subscription']);
            Route::post('/subscription/upgrade', [ProfessionalController::class, 'upgradeSubscription']);
        });

        // Client routes
        Route::prefix('client')->name('api.client.')->group(function () {
            Route::get('/dashboard', [ClientController::class, 'dashboard']);
            Route::get('/profile', [ClientController::class, 'profile']);
            Route::put('/profile', [ClientController::class, 'updateProfile']);
            Route::get('/professionals', [ClientController::class, 'searchProfessionals']);
            Route::get('/consultations', [ClientController::class, 'consultations']);
            Route::post('/book/{professional}', [ClientController::class, 'storeBooking']);
        });

        // Consultation routes
        Route::prefix('consultations')->name('api.consultation.')->group(function () {
            Route::get('/{consultation}', [ConsultationController::class, 'show']);
            Route::post('/{consultation}/start', [ConsultationController::class, 'start']);
            Route::post('/{consultation}/end', [ConsultationController::class, 'end']);
            Route::get('/{consultation}/meeting', [ConsultationController::class, 'joinMeeting']);
        });

        // Payment routes
        Route::prefix('payments')->name('api.payment.')->group(function () {
            Route::post('/consultation/{consultation}', [PaymentController::class, 'processConsultationPayment']);
            Route::post('/subscription/{subscription}', [PaymentController::class, 'processSubscriptionPayment']);
            Route::get('/history', [PaymentController::class, 'paymentHistory']);
            Route::get('/invoice/{payment}', [PaymentController::class, 'downloadInvoice']);
            Route::post('/{payment}/refund', [PaymentController::class, 'requestRefund']);
        });

        // Review routes
        Route::prefix('reviews')->name('api.review.')->group(function () {
            Route::post('/consultation/{consultation}', [ReviewController::class, 'store']);
            Route::get('/my-reviews', [ReviewController::class, 'myReviews']);
            Route::get('/received', [ReviewController::class, 'receivedReviews']);
            Route::get('/professional/{professional}', [ReviewController::class, 'professionalReviews']);
            Route::put('/{review}', [ReviewController::class, 'update']);
            Route::post('/{review}/report', [ReviewController::class, 'report']);
        });
    });
});