@extends('layouts.app')

@section('title', 'Reviews Received')

@section('content')
<div class="container mt-4">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2>
                    <i class="fas fa-star text-warning me-2"></i>
                    Reviews Received
                </h2>
                
                <!-- Filter Dropdown -->
                <div class="dropdown">
                    <button class="btn btn-outline-primary dropdown-toggle" type="button" 
                            data-bs-toggle="dropdown">
                        <i class="fas fa-filter me-1"></i>
                        Filter by Rating
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="{{ route('review.received') }}">All Reviews</a></li>
                        <li><hr class="dropdown-divider"></li>
                        @for($i = 5; $i >= 1; $i--)
                            <li>
                                <a class="dropdown-item {{ request('rating') == $i ? 'active' : '' }}" 
                                   href="{{ request()->fullUrlWithQuery(['rating' => $i]) }}">
                                    {{ $i }} Star{{ $i > 1 ? 's' : '' }}
                                </a>
                            </li>
                        @endfor
                    </ul>
                </div>
            </div>

            @if($reviews->count() > 0)
                <!-- Reviews Summary -->
                @php
                    $averageRating = $reviews->avg('rating');
                    $totalReviews = $reviews->total();
                    $positiveReviews = $reviews->where('rating', '>=', 4)->count();
                    $recentReviews = $reviews->where('created_at', '>=', now()->subDays(30))->count();
                @endphp

                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card text-center bg-primary text-white">
                            <div class="card-body">
                                <h3 class="mb-1">{{ $totalReviews }}</h3>
                                <small>Total Reviews</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center bg-success text-white">
                            <div class="card-body">
                                <h3 class="mb-1">{{ number_format($averageRating, 1) }}</h3>
                                <div class="star-display text-white">
                                    @for($i = 1; $i <= 5; $i++)
                                        <i class="fas fa-star {{ $i <= round($averageRating) ? '' : 'text-light' }}"></i>
                                    @endfor
                                </div>
                                <small>Average Rating</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center bg-warning text-white">
                            <div class="card-body">
                                <h3 class="mb-1">{{ $positiveReviews }}</h3>
                                <small>Positive Reviews</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center bg-info text-white">
                            <div class="card-body">
                                <h3 class="mb-1">{{ $recentReviews }}</h3>
                                <small>This Month</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Reviews List -->
                <div class="reviews-list">
                    @foreach($reviews as $review)
                        <div class="card mb-3 {{ $review->rating >= 4 ? 'border-success' : ($review->rating <= 2 ? 'border-warning' : '') }}">
                            <div class="card-body">
                                <!-- Review Header -->
                                <div class="d-flex justify-content-between align-items-start mb-3">
                                    <div class="d-flex align-items-center">
                                        <div class="me-3">
                                            @if($review->is_anonymous)
                                                <div class="bg-secondary rounded-circle d-inline-flex align-items-center justify-content-center" 
                                                     style="width: 50px; height: 50px;">
                                                    <i class="fas fa-user text-white"></i>
                                                </div>
                                            @else
                                                <div class="bg-primary rounded-circle d-inline-flex align-items-center justify-content-center" 
                                                     style="width: 50px; height: 50px;">
                                                    <span class="text-white fs-5">{{ substr($review->reviewer->name, 0, 1) }}</span>
                                                </div>
                                            @endif
                                        </div>
                                        <div>
                                            <h6 class="mb-1">
                                                {{ $review->is_anonymous ? 'Anonymous ' . ucfirst($review->reviewer->user_type) : $review->reviewer->name }}
                                            </h6>
                                            <div class="star-display">
                                                @for($i = 1; $i <= 5; $i++)
                                                    <i class="fas fa-star {{ $i <= $review->rating ? 'text-warning' : 'text-light' }}"></i>
                                                @endfor
                                                <span class="text-muted ms-2">{{ $review->created_at->format('M d, Y') }}</span>
                                            </div>
                                            <small class="text-muted">
                                                {{ $review->reviewer->user_type === 'client' ? 'Client' : 'Professional' }}
                                            </small>
                                        </div>
                                    </div>
                                    
                                    <!-- Rating Badge -->
                                    <span class="badge bg-{{ $review->rating >= 4 ? 'success' : ($review->rating >= 3 ? 'warning' : 'danger') }} fs-6">
                                        {{ $review->rating }}/5
                                    </span>
                                </div>

                                <!-- Review Content -->
                                <div class="review-content mb-3">
                                    <p class="mb-0 fs-6">{{ $review->comment }}</p>
                                </div>

                                <!-- Consultation Context -->
                                <div class="consultation-context mb-3 p-3 bg-light rounded">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-calendar me-1"></i>
                                                <strong>Consultation Date:</strong> {{ $review->consultation->scheduled_at->format('M d, Y') }}
                                            </small>
                                        </div>
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-clock me-1"></i>
                                                <strong>Duration:</strong> {{ $review->consultation->duration }} minutes
                                            </small>
                                        </div>
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-video me-1"></i>
                                                <strong>Type:</strong> {{ ucfirst($review->consultation->consultation_type) }}
                                            </small>
                                        </div>
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-check-circle me-1 text-success"></i>
                                                <strong>Status:</strong> {{ ucfirst($review->consultation->status) }}
                                            </small>
                                        </div>
                                    </div>
                                </div>

                                <!-- Review Actions -->
                                <div class="d-flex justify-content-between align-items-center">
                                    <div class="review-tags">
                                        <span class="badge bg-{{ $review->status === 'active' ? 'success' : 'warning' }}">
                                            {{ ucfirst($review->status) }}
                                        </span>
                                        @if($review->is_anonymous)
                                            <span class="badge bg-secondary ms-1">Anonymous</span>
                                        @endif
                                    </div>
                                    
                                    <div class="review-actions">
                                        <a href="{{ route('review.show', $review) }}" class="btn btn-outline-primary btn-sm">
                                            <i class="fas fa-eye me-1"></i>View Details
                                        </a>
                                        <a href="{{ route('consultation.show', $review->consultation) }}" class="btn btn-outline-secondary btn-sm">
                                            <i class="fas fa-calendar me-1"></i>View Consultation
                                        </a>
                                    </div>
                                </div>

                                <!-- Response Option (for professionals) -->
                                @if(Auth::user()->user_type === 'professional' && $review->reviewer->user_type === 'client')
                                    <div class="mt-3 pt-3 border-top">
                                        <button class="btn btn-outline-info btn-sm" onclick="respondToReview({{ $review->id }})">
                                            <i class="fas fa-reply me-1"></i>Respond to Review
                                        </button>
                                        <small class="text-muted ms-2">Thank your client or address their concerns</small>
                                    </div>
                                @endif
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-center">
                    {{ $reviews->links() }}
                </div>

                <!-- Insights Section -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6><i class="fas fa-chart-line text-success me-2"></i>Review Insights</h6>
                                <ul class="mb-0 small">
                                    <li>Your average rating is {{ number_format($averageRating, 1) }}/5.0</li>
                                    <li>{{ number_format(($positiveReviews / $totalReviews) * 100, 1) }}% of reviews are positive (4+ stars)</li>
                                    <li>You've received {{ $recentReviews }} reviews this month</li>
                                    <li>Keep up the great work to maintain high ratings!</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6><i class="fas fa-lightbulb text-warning me-2"></i>Improvement Tips</h6>
                                <ul class="mb-0 small">
                                    <li>Respond to reviews to show you care about feedback</li>
                                    <li>Address negative feedback constructively</li>
                                    <li>Use reviews to identify areas for improvement</li>
                                    <li>Encourage satisfied clients to leave reviews</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            @else
                <!-- No Reviews -->
                <div class="text-center py-5">
                    <i class="fas fa-star fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">No reviews received yet</h5>
                    <p class="text-muted">
                        You haven't received any reviews yet. Complete more consultations to start receiving feedback from clients.
                    </p>
                    
                    @if(Auth::user()->user_type === 'professional')
                        <a href="{{ route('professional.dashboard') }}" class="btn btn-primary">
                            <i class="fas fa-tachometer-alt me-1"></i>Go to Dashboard
                        </a>
                    @endif
                </div>
            @endif
        </div>
    </div>
</div>

@push('styles')
<style>
.star-display {
    font-size: 0.875rem;
}

.review-content {
    line-height: 1.6;
}

.consultation-context {
    background-color: #f8f9fa;
}

.card {
    transition: box-shadow 0.2s;
}

.card:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.border-success {
    border-left: 4px solid #198754 !important;
}

.border-warning {
    border-left: 4px solid #ffc107 !important;
}

.badge {
    font-size: 0.7em;
}
</style>
@endpush

@push('scripts')
<script>
function respondToReview(reviewId) {
    // In a real application, this would open a modal for responding to reviews
    alert('Response functionality would open here for review ID: ' + reviewId);
}
</script>
@endpush
@endsection