@extends('layouts.app')

@section('title', 'Payment History')

@section('content')
<div class="container mt-4">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2>Payment History</h2>
                <div class="d-flex gap-2">
                    <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#filterModal">
                        <i class="fas fa-filter me-2"></i>Filter
                    </button>
                    <a href="{{ route('payment.export') }}" class="btn btn-outline-success">
                        <i class="fas fa-download me-2"></i>Export
                    </a>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card text-center">
                        <div class="card-body">
                            <h5 class="text-success">₹{{ number_format($payments->where('status', 'completed')->sum('total_amount'), 2) }}</h5>
                            <small class="text-muted">Total Paid</small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-center">
                        <div class="card-body">
                            <h5 class="text-primary">{{ $payments->where('status', 'completed')->count() }}</h5>
                            <small class="text-muted">Successful Payments</small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-center">
                        <div class="card-body">
                            <h5 class="text-warning">{{ $payments->where('status', 'pending')->count() }}</h5>
                            <small class="text-muted">Pending Payments</small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-center">
                        <div class="card-body">
                            <h5 class="text-danger">{{ $payments->where('status', 'failed')->count() }}</h5>
                            <small class="text-muted">Failed Payments</small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Payments Table -->
            <div class="card">
                <div class="card-body">
                    @if($payments->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Type</th>
                                        <th>Description</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($payments as $payment)
                                    <tr>
                                        <td>
                                            <div>
                                                <strong>{{ $payment->created_at->format('d M Y') }}</strong>
                                                <br>
                                                <small class="text-muted">{{ $payment->created_at->format('h:i A') }}</small>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge bg-{{ $payment->payment_type == 'consultation' ? 'info' : 'success' }}">
                                                {{ ucfirst($payment->payment_type) }}
                                            </span>
                                        </td>
                                        <td>
                                            @if($payment->payable_type == 'App\\Models\\Consultation')
                                                <div>
                                                    <strong>{{ $payment->payable->professional->user->name }}</strong>
                                                    <br>
                                                    <small class="text-muted">
                                                        {{ $payment->payable->scheduled_at->format('d M Y, h:i A') }}
                                                    </small>
                                                </div>
                                            @elseif($payment->payable_type == 'App\\Models\\Subscription')
                                                <div>
                                                    <strong>{{ $payment->payable->subscriptionPlan->name }} Plan</strong>
                                                    <br>
                                                    <small class="text-muted">
                                                        {{ $payment->payable->start_date->format('M Y') }} subscription
                                                    </small>
                                                </div>
                                            @endif
                                        </td>
                                        <td>
                                            <div>
                                                <strong>₹{{ number_format($payment->total_amount, 2) }}</strong>
                                                @if($payment->gst_amount > 0)
                                                    <br>
                                                    <small class="text-muted">
                                                        (₹{{ number_format($payment->base_amount, 2) }} + GST)
                                                    </small>
                                                @endif
                                            </div>
                                        </td>
                                        <td>
                                            @php
                                                $statusColor = match($payment->status) {
                                                    'completed' => 'success',
                                                    'pending' => 'warning',
                                                    'failed' => 'danger',
                                                    'refunded' => 'info',
                                                    default => 'secondary'
                                                };
                                            @endphp
                                            <span class="badge bg-{{ $statusColor }}">
                                                {{ ucfirst($payment->status) }}
                                            </span>
                                            @if($payment->status == 'completed' && $payment->created_at->diffInDays() <= 7)
                                                <br>
                                                <small class="text-success">
                                                    <i class="fas fa-shield-alt me-1"></i>Protected
                                                </small>
                                            @endif
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm" role="group">
                                                @if($payment->status == 'completed')
                                                    <a href="{{ route('payment.invoice', $payment) }}" 
                                                       class="btn btn-outline-primary" 
                                                       title="Download Invoice">
                                                        <i class="fas fa-file-invoice"></i>
                                                    </a>
                                                @endif
                                                
                                                @if($payment->status == 'completed' && $payment->payment_type == 'consultation')
                                                    @php
                                                        $consultation = $payment->payable;
                                                        $canRefund = $consultation && $consultation->scheduled_at > now()->addHours(2);
                                                    @endphp
                                                    @if($canRefund)
                                                        <button class="btn btn-outline-danger" 
                                                                onclick="showRefundModal('{{ $payment->payment_id }}')"
                                                                title="Request Refund">
                                                            <i class="fas fa-undo"></i>
                                                        </button>
                                                    @endif
                                                @endif

                                                <button class="btn btn-outline-info" 
                                                        onclick="showPaymentDetails('{{ $payment->payment_id }}')"
                                                        title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <div class="d-flex justify-content-center mt-4">
                            {{ $payments->links() }}
                        </div>
                    @else
                        <div class="text-center py-5">
                            <i class="fas fa-credit-card fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No payments found</h5>
                            <p class="text-muted">Your payment history will appear here once you make your first payment.</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Filter Modal -->
<div class="modal fade" id="filterModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="GET" action="{{ route('payment.history') }}">
                <div class="modal-header">
                    <h5 class="modal-title">Filter Payments</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select">
                            <option value="">All Statuses</option>
                            <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                            <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                            <option value="failed" {{ request('status') == 'failed' ? 'selected' : '' }}>Failed</option>
                            <option value="refunded" {{ request('status') == 'refunded' ? 'selected' : '' }}>Refunded</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Payment Type</label>
                        <select name="type" class="form-select">
                            <option value="">All Types</option>
                            <option value="consultation" {{ request('type') == 'consultation' ? 'selected' : '' }}>Consultation</option>
                            <option value="subscription" {{ request('type') == 'subscription' ? 'selected' : '' }}>Subscription</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Apply Filter</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Refund Modal -->
<div class="modal fade" id="refundModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="refundForm" method="POST">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Request Refund</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <strong>Refund Policy:</strong> Refunds are only available up to 2 hours before the scheduled consultation.
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Reason for Refund</label>
                        <textarea name="reason" class="form-control" rows="3" required 
                                  placeholder="Please provide a reason for requesting the refund..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Request Refund</button>
                </div>
            </form>
        </div>
    </div>
</div>

@push('scripts')
<script>
function showRefundModal(paymentId) {
    const modal = new bootstrap.Modal(document.getElementById('refundModal'));
    document.getElementById('refundForm').action = `/payment/${paymentId}/refund`;
    modal.show();
}

function showPaymentDetails(paymentId) {
    // In a real application, this would show a modal with detailed payment information
    alert('Payment details modal would open here. Payment ID: ' + paymentId);
}
</script>
@endpush
@endsection