<?php

namespace App\Services;

use App\Models\Consultation;
use App\Models\Professional;
use App\Models\Client;
use App\Models\Payment;
use App\Models\Commission;
use Carbon\Carbon;

class ConsultationService
{
    /**
     * Create a new consultation booking
     */
    public function createConsultation(Client $client, Professional $professional, $data)
    {
        // Calculate fees
        $consultationFee = $professional->consultation_fee;
        $platformFee = $this->calculatePlatformFee($consultationFee);
        $gstAmount = $this->calculateGST($consultationFee + $platformFee);
        $totalAmount = $consultationFee + $platformFee + $gstAmount;

        // Create consultation
        $consultation = Consultation::create([
            'consultation_id' => $this->generateConsultationId(),
            'client_id' => $client->id,
            'professional_id' => $professional->id,
            'consultation_type' => $data['consultation_type'],
            'scheduled_at' => $data['scheduled_at'],
            'status' => 'scheduled',
            'consultation_fee' => $consultationFee,
            'platform_fee' => $platformFee,
            'gst_amount' => $gstAmount,
            'total_amount' => $totalAmount,
            'meeting_id' => $this->generateMeetingId(),
            'client_notes' => $data['notes'] ?? null,
        ]);

        // Generate Jitsi Meet URL
        $consultation->update([
            'meeting_url' => $this->generateJitsiMeetUrl($consultation->meeting_id),
        ]);

        return $consultation;
    }

    /**
     * Calculate platform fee (typically fixed amount or percentage)
     */
    private function calculatePlatformFee($consultationFee)
    {
        // Fixed platform fee of ₹100 or 5% of consultation fee, whichever is higher
        $percentageFee = $consultationFee * 0.05;
        return max(100, $percentageFee);
    }

    /**
     * Calculate GST (18% on total amount)
     */
    private function calculateGST($amount)
    {
        return round(($amount * 18) / 100, 2);
    }

    /**
     * Generate unique consultation ID
     */
    private function generateConsultationId()
    {
        return 'CON-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -6));
    }

    /**
     * Generate unique meeting ID for Jitsi
     */
    private function generateMeetingId()
    {
        return 'medex-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -8));
    }

    /**
     * Generate Jitsi Meet URL
     */
    private function generateJitsiMeetUrl($meetingId)
    {
        return "https://meet.jit.si/{$meetingId}";
    }

    /**
     * Start a consultation
     */
    public function startConsultation(Consultation $consultation)
    {
        $consultation->update([
            'status' => 'in_progress',
            'started_at' => now(),
        ]);

        return $consultation;
    }

    /**
     * End a consultation
     */
    public function endConsultation(Consultation $consultation, $notes = null)
    {
        $duration = now()->diffInMinutes($consultation->started_at);
        
        $consultation->update([
            'status' => 'completed',
            'ended_at' => now(),
            'duration_minutes' => $duration,
            'consultation_notes' => $notes,
        ]);

        // Calculate and create commission
        $this->calculateCommission($consultation);

        return $consultation;
    }

    /**
     * Cancel a consultation
     */
    public function cancelConsultation(Consultation $consultation, $reason = null, $cancelledBy = null)
    {
        $consultation->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => $reason,
            'cancelled_by' => $cancelledBy,
        ]);

        // Handle refund if payment was made
        $payment = $consultation->payment;
        if ($payment && $payment->status === 'completed') {
            // Process refund logic here
            $this->processRefund($payment, $reason);
        }

        return $consultation;
    }

    /**
     * Calculate commission for completed consultation
     */
    private function calculateCommission(Consultation $consultation)
    {
        $professional = $consultation->professional;
        $subscription = $professional->activeSubscription();
        
        if (!$subscription) {
            $commissionPercentage = 50; // Default commission
        } else {
            $commissionPercentage = $subscription->subscriptionPlan->commission_percentage;
        }

        $commissionData = Commission::calculateCommission(
            $consultation->consultation_fee,
            $commissionPercentage,
            $consultation->platform_fee
        );

        Commission::create([
            'consultation_id' => $consultation->id,
            'professional_id' => $professional->id,
            'consultation_fee' => $consultation->consultation_fee,
            'commission_percentage' => $commissionPercentage,
            'platform_commission' => $commissionData['platform_commission'],
            'professional_earning' => $commissionData['professional_earning'],
            'platform_fee' => $consultation->platform_fee,
            'gst_amount' => $consultation->gst_amount,
            'status' => 'completed',
            'processed_at' => now(),
        ]);
    }

    /**
     * Process refund for cancelled consultation
     */
    private function processRefund(Payment $payment, $reason)
    {
        // Calculate refund amount based on cancellation policy
        $refundAmount = $this->calculateRefundAmount($payment);
        
        $payment->update([
            'status' => 'refunded',
            'refunded_at' => now(),
            'refund_amount' => $refundAmount,
            'refund_reason' => $reason,
        ]);

        // Here you would integrate with payment gateway to process actual refund
        // For now, we just mark it as refunded in the system
    }

    /**
     * Calculate refund amount based on cancellation policy
     */
    private function calculateRefundAmount(Payment $payment)
    {
        // Simple refund policy: full refund if cancelled 24 hours before
        // 50% refund if cancelled within 24 hours
        // No refund if cancelled within 2 hours
        
        $consultation = $payment->payable;
        $hoursUntilConsultation = now()->diffInHours($consultation->scheduled_at);
        
        if ($hoursUntilConsultation >= 24) {
            return $payment->total_amount; // Full refund
        } elseif ($hoursUntilConsultation >= 2) {
            return $payment->total_amount * 0.5; // 50% refund
        } else {
            return 0; // No refund
        }
    }

    /**
     * Get available time slots for a professional
     */
    public function getAvailableSlots(Professional $professional, $date)
    {
        $availability = $professional->availability ?? [
            'monday' => ['09:00', '17:00'],
            'tuesday' => ['09:00', '17:00'],
            'wednesday' => ['09:00', '17:00'],
            'thursday' => ['09:00', '17:00'],
            'friday' => ['09:00', '17:00'],
            'saturday' => ['09:00', '13:00'],
            'sunday' => null, // Closed
        ];

        $dayOfWeek = strtolower(Carbon::parse($date)->format('l'));
        
        if (!isset($availability[$dayOfWeek]) || !$availability[$dayOfWeek]) {
            return []; // Professional not available on this day
        }

        [$startTime, $endTime] = $availability[$dayOfWeek];
        
        // Get existing bookings for the date
        $existingBookings = $professional->consultations()
            ->whereDate('scheduled_at', $date)
            ->whereIn('status', ['scheduled', 'in_progress'])
            ->pluck('scheduled_at')
            ->map(function ($datetime) {
                return Carbon::parse($datetime)->format('H:i');
            })
            ->toArray();

        // Generate 30-minute slots
        $slots = [];
        $current = Carbon::parse($date . ' ' . $startTime);
        $end = Carbon::parse($date . ' ' . $endTime);

        while ($current < $end) {
            $timeSlot = $current->format('H:i');
            
            // Check if slot is available
            if (!in_array($timeSlot, $existingBookings) && $current > now()) {
                $slots[] = [
                    'time' => $timeSlot,
                    'datetime' => $current->toDateTimeString(),
                    'formatted' => $current->format('g:i A'),
                ];
            }
            
            $current->addMinutes(30);
        }

        return $slots;
    }

    /**
     * Get consultation statistics for dashboard
     */
    public function getConsultationStats($userId, $userType)
    {
        if ($userType === 'professional') {
            $professional = Professional::where('user_id', $userId)->first();
            if (!$professional) return [];

            return [
                'total' => $professional->consultations()->count(),
                'completed' => $professional->consultations()->where('status', 'completed')->count(),
                'scheduled' => $professional->consultations()->where('status', 'scheduled')->count(),
                'cancelled' => $professional->consultations()->where('status', 'cancelled')->count(),
                'this_month' => $professional->consultations()->whereMonth('created_at', now()->month)->count(),
            ];
        } else {
            $client = Client::where('user_id', $userId)->first();
            if (!$client) return [];

            return [
                'total' => $client->consultations()->count(),
                'completed' => $client->consultations()->where('status', 'completed')->count(),
                'scheduled' => $client->consultations()->where('status', 'scheduled')->count(),
                'cancelled' => $client->consultations()->where('status', 'cancelled')->count(),
                'this_month' => $client->consultations()->whereMonth('created_at', now()->month)->count(),
            ];
        }
    }
}