<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Review extends Model
{
    use HasFactory;

    protected $fillable = [
        'consultation_id',
        'from_user_id',
        'to_user_id',
        'role_from',
        'rating',
        'comment',
        'criteria_ratings',
        'status',
        'approved_by_admin',
        'approved_by',
        'approved_at',
        'is_flagged',
        'flag_reason',
    ];

    protected $casts = [
        'criteria_ratings' => 'array',
        'approved_by_admin' => 'boolean',
        'approved_at' => 'datetime',
        'is_flagged' => 'boolean',
    ];

    /**
     * Get the consultation this review belongs to
     */
    public function consultation()
    {
        return $this->belongsTo(Consultation::class);
    }

    /**
     * Get the user who wrote the review
     */
    public function fromUser()
    {
        return $this->belongsTo(User::class, 'from_user_id');
    }

    /**
     * Get the user who received the review
     */
    public function toUser()
    {
        return $this->belongsTo(User::class, 'to_user_id');
    }

    /**
     * Get the admin who approved the review
     */
    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Check if review is approved
     */
    public function isApproved()
    {
        return $this->approved_by_admin;
    }

    /**
     * Check if review is from client to professional
     */
    public function isClientReview()
    {
        return $this->role_from === 'client';
    }

    /**
     * Check if review is from professional to client
     */
    public function isProfessionalReview()
    {
        return $this->role_from === 'professional';
    }

    /**
     * Scope for approved reviews
     */
    public function scopeApproved($query)
    {
        return $query->where('approved_by_admin', true);
    }

    /**
     * Scope for pending reviews
     */
    public function scopePending($query)
    {
        return $query->where('approved_by_admin', false);
    }

    /**
     * Scope for client reviews
     */
    public function scopeFromClients($query)
    {
        return $query->where('role_from', 'client');
    }

    /**
     * Scope for professional reviews
     */
    public function scopeFromProfessionals($query)
    {
        return $query->where('role_from', 'professional');
    }

    /**
     * Calculate average rating for a user
     */
    public static function getAverageRating($userId, $roleFrom = null)
    {
        $query = self::where('to_user_id', $userId)->where('approved_by_admin', true);
        
        if ($roleFrom) {
            $query->where('role_from', $roleFrom);
        }
        
        return $query->avg('rating') ?? 0;
    }
}
