<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Commission extends Model
{
    use HasFactory;

    protected $fillable = [
        'consultation_id',
        'professional_id',
        'consultation_fee',
        'commission_percentage',
        'platform_commission',
        'professional_earning',
        'platform_fee',
        'gst_amount',
        'status',
        'processed_at',
    ];

    protected $casts = [
        'consultation_fee' => 'decimal:2',
        'platform_commission' => 'decimal:2',
        'professional_earning' => 'decimal:2',
        'platform_fee' => 'decimal:2',
        'gst_amount' => 'decimal:2',
        'processed_at' => 'datetime',
    ];

    /**
     * Get the consultation that this commission belongs to
     */
    public function consultation()
    {
        return $this->belongsTo(Consultation::class);
    }

    /**
     * Get the professional that earned this commission
     */
    public function professional()
    {
        return $this->belongsTo(Professional::class);
    }

    /**
     * Get the payout that includes this commission
     */
    public function payout()
    {
        return $this->belongsTo(Payout::class);
    }

    /**
     * Calculate commission amounts
     */
    public static function calculateCommission($consultationFee, $commissionPercentage, $platformFee = 0, $gstRate = 18)
    {
        $platformCommission = ($consultationFee * $commissionPercentage) / 100;
        $professionalEarning = $consultationFee - $platformCommission;
        $gstAmount = (($consultationFee + $platformFee) * $gstRate) / 100;

        return [
            'platform_commission' => round($platformCommission, 2),
            'professional_earning' => round($professionalEarning, 2),
            'gst_amount' => round($gstAmount, 2),
            'total_client_payment' => round($consultationFee + $platformFee + $gstAmount, 2),
        ];
    }

    /**
     * Scope for pending commissions
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for completed commissions
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}
