<?php

namespace App\Http\Controllers;

use App\Models\Professional;
use App\Models\SubscriptionPlan;
use App\Models\Consultation;
use App\Services\SubscriptionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProfessionalController extends Controller
{
    protected $subscriptionService;

    public function __construct(SubscriptionService $subscriptionService)
    {
        $this->middleware('auth');
        $this->subscriptionService = $subscriptionService;
    }

    /**
     * Show professional dashboard
     */
    public function dashboard()
    {
        $user = Auth::user();
        $professional = $user->professional;

        if (!$professional) {
            return redirect()->route('home')->with('error', 'Professional profile not found.');
        }

        // Comprehensive statistics
        $stats = [
            'total_consultations' => $professional->consultations()->count(),
            'consultations_this_month' => $professional->consultations()
                ->whereMonth('scheduled_at', now()->month)
                ->count(),
            'consultations_growth' => $this->calculateGrowthRate(
                $professional->consultations()->whereMonth('scheduled_at', now()->subMonth()->month)->count(),
                $professional->consultations()->whereMonth('scheduled_at', now()->month)->count()
            ),
            'total_earnings' => $professional->commissions()->where('status', 'completed')->sum('professional_earning'),
            'earnings_this_month' => $professional->commissions()
                ->where('status', 'completed')
                ->whereMonth('created_at', now()->month)
                ->sum('professional_earning'),
            'average_rating' => $professional->average_rating ?? 0,
            'total_reviews' => $professional->total_reviews ?? 0,
            'pending_reviews' => $professional->consultations()
                ->where('status', 'completed')
                ->whereDoesntHave('reviews', function($q) use ($professional) {
                    $q->where('reviewee_id', $professional->user_id);
                })
                ->count(),
            'reviews_this_month' => $professional->reviews()
                ->whereMonth('created_at', now()->month)
                ->count(),
            'response_time' => $this->calculateAverageResponseTime($professional),
        ];

        // Upcoming consultations
        $upcomingConsultations = $professional->consultations()
            ->with(['client.user'])
            ->where('status', 'scheduled')
            ->where('scheduled_at', '>=', now())
            ->orderBy('scheduled_at')
            ->limit(5)
            ->get();

        // Recent reviews
        $recentReviews = $professional->reviews()
            ->with(['reviewer'])
            ->where('status', 'active')
            ->latest()
            ->limit(5)
            ->get();

        // Profile completion analysis
        $profileCompletion = $this->calculateProfileCompletion($professional);

        // Active subscription
        $subscription = $professional->activeSubscription();

        return view('professional.dashboard', compact(
            'professional',
            'stats',
            'upcomingConsultations',
            'recentReviews',
            'profileCompletion',
            'subscription'
        ));
    }

    /**
     * Calculate growth rate percentage
     */
    private function calculateGrowthRate($previous, $current)
    {
        if ($previous == 0) return $current > 0 ? 100 : 0;
        return round((($current - $previous) / $previous) * 100, 1);
    }

    /**
     * Calculate average response time
     */
    private function calculateAverageResponseTime($professional)
    {
        // Mock implementation - in real scenario, track response times
        return '2h';
    }

    /**
     * Calculate profile completion percentage
     */
    private function calculateProfileCompletion($professional)
    {
        $fields = [
            'profile_photo' => $professional->profile_photo,
            'bio' => $professional->bio,
            'specialization' => $professional->specialization,
            'experience_years' => $professional->experience_years,
            'education' => $professional->education,
            'certifications' => $professional->certifications,
            'consultation_fee' => $professional->consultation_fee,
        ];

        $completed = collect($fields)->filter()->count();
        $total = count($fields);
        $percentage = round(($completed / $total) * 100);

        $missing = collect($fields)->filter(function($value, $key) {
            return empty($value);
        })->keys()->toArray();

        return [
            'percentage' => $percentage,
            'completed' => $completed,
            'total' => $total,
            'missing' => $missing
        ];
    }

    /**
     * Show professional profile
     */
    public function profile()
    {
        $user = Auth::user();
        $professional = $user->professional;

        return view('professional.profile', compact('professional'));
    }

    /**
     * Update professional profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        $professional = $user->professional;

        $request->validate([
            'consultation_fee' => 'required|numeric|min:0',
            'experience_years' => 'required|integer|min:0',
            'specializations' => 'nullable|array',
            'office_address' => 'nullable|string',
            'bio' => 'nullable|string|max:1000',
        ]);

        $professional->update([
            'consultation_fee' => $request->consultation_fee,
            'experience_years' => $request->experience_years,
            'specializations' => $request->specializations,
            'office_address' => $request->office_address,
            'bio' => $request->bio,
        ]);

        return back()->with('success', 'Profile updated successfully!');
    }

    /**
     * Show consultations
     */
    public function consultations(Request $request)
    {
        $user = Auth::user();
        $professional = $user->professional;

        $status = $request->get('status', 'all');
        
        $consultationsQuery = $professional->consultations()->with(['client.user']);

        if ($status !== 'all') {
            $consultationsQuery->where('status', $status);
        }

        $consultations = $consultationsQuery->orderBy('scheduled_at', 'desc')->paginate(10);

        return view('professional.consultations', compact('consultations', 'status'));
    }

    /**
     * Show earnings
     */
    public function earnings()
    {
        $user = Auth::user();
        $professional = $user->professional;

        $totalEarnings = $professional->commissions()->where('status', 'completed')->sum('professional_earning');
        $pendingEarnings = $professional->commissions()->where('status', 'pending')->sum('professional_earning');
        $thisMonthEarnings = $professional->commissions()
            ->where('status', 'completed')
            ->whereMonth('created_at', now()->month)
            ->sum('professional_earning');

        $recentEarnings = $professional->commissions()
            ->with(['consultation.client.user'])
            ->latest()
            ->limit(10)
            ->get();

        return view('professional.earnings', compact(
            'totalEarnings',
            'pendingEarnings',
            'thisMonthEarnings',
            'recentEarnings'
        ));
    }

    /**
     * Show subscription page
     */
    public function subscription()
    {
        $user = Auth::user();
        $professional = $user->professional;
        
        $currentSubscription = $professional->activeSubscription();
        $subscriptionPlans = SubscriptionPlan::active()->orderBy('sort_order')->get();

        return view('professional.subscription', compact('currentSubscription', 'subscriptionPlans'));
    }

    /**
     * Upgrade subscription
     */
    public function upgradeSubscription(Request $request)
    {
        $user = Auth::user();
        $professional = $user->professional;

        $request->validate([
            'plan_id' => 'required|exists:subscription_plans,id',
            'billing_cycle' => 'required|in:monthly,yearly',
        ]);

        $plan = SubscriptionPlan::findOrFail($request->plan_id);

        try {
            $subscription = $this->subscriptionService->upgradeSubscription(
                $professional,
                $plan,
                $request->billing_cycle
            );

            return back()->with('success', 'Subscription upgraded successfully!');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to upgrade subscription. Please try again.');
        }
    }
}
