<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Consultation;
use App\Models\Subscription;
use App\Services\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PaymentController extends Controller
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->middleware('auth');
        $this->paymentService = $paymentService;
    }

    /**
     * Show payment page for consultation
     */
    public function showConsultationPayment(Consultation $consultation)
    {
        $user = Auth::user();
        
        if ($user->id !== $consultation->client->user_id) {
            abort(403, 'Unauthorized access to payment.');
        }

        if ($consultation->status !== 'scheduled') {
            return redirect()->route('client.consultations')
                           ->with('error', 'Payment not available for this consultation.');
        }

        // Check if payment already exists
        $existingPayment = $consultation->payment;
        if ($existingPayment && $existingPayment->status === 'completed') {
            return redirect()->route('consultation.show', $consultation)
                           ->with('info', 'Payment already completed for this consultation.');
        }

        try {
            $payment = $this->paymentService->createPaymentIntent(
                $consultation,
                $user->id,
                'consultation'
            );

            return view('payment.consultation', compact('consultation', 'payment'));
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to create payment. Please try again.');
        }
    }

    /**
     * Show payment page for subscription
     */
    public function showSubscriptionPayment(Subscription $subscription)
    {
        $user = Auth::user();
        
        if ($user->id !== $subscription->professional->user_id) {
            abort(403, 'Unauthorized access to payment.');
        }

        try {
            $payment = $this->paymentService->createPaymentIntent(
                $subscription,
                $user->id,
                'subscription'
            );

            return view('payment.subscription', compact('subscription', 'payment'));
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to create payment. Please try again.');
        }
    }

    /**
     * Process payment success callback
     */
    public function paymentSuccess(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,payment_id',
            'gateway_payment_id' => 'required|string',
            'gateway_signature' => 'nullable|string',
        ]);

        $payment = Payment::where('payment_id', $request->payment_id)->first();
        
        if (!$payment) {
            return redirect()->route('home')->with('error', 'Payment not found.');
        }

        // Verify payment signature (in production, this should be properly validated)
        $this->verifyPaymentSignature($payment, $request->gateway_payment_id, $request->gateway_signature);

        try {
            $this->paymentService->processPaymentSuccess(
                $payment,
                $request->gateway_payment_id,
                $request->all()
            );

            if ($payment->payable instanceof Consultation) {
                return redirect()->route('consultation.show', $payment->payable)
                               ->with('success', 'Payment successful! Your consultation is confirmed.');
            } else {
                return redirect()->route('professional.subscription')
                               ->with('success', 'Payment successful! Your subscription is now active.');
            }
        } catch (\Exception $e) {
            return redirect()->route('home')->with('error', 'Payment processing failed.');
        }
    }

    /**
     * Process payment failure callback
     */
    public function paymentFailure(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,payment_id',
            'error_description' => 'nullable|string',
        ]);

        $payment = Payment::where('payment_id', $request->payment_id)->first();
        
        if ($payment) {
            $this->paymentService->processPaymentFailure(
                $payment,
                $request->error_description,
                $request->all()
            );
        }

        return redirect()->route('home')->with('error', 'Payment failed. Please try again.');
    }

    /**
     * Download invoice
     */
    public function downloadInvoice(Payment $payment)
    {
        $user = Auth::user();
        
        // Check access rights
        if ($user->id !== $payment->user_id && $user->user_type !== 'admin') {
            abort(403, 'Unauthorized access to invoice.');
        }

        if ($payment->status !== 'completed') {
            return back()->with('error', 'Invoice not available for incomplete payments.');
        }

        $invoiceData = $this->paymentService->generateInvoiceData($payment);

        return view('payment.invoice', compact('payment', 'invoiceData'));
    }

    /**
     * Verify payment signature (mock implementation)
     */
    private function verifyPaymentSignature($payment, $gatewayPaymentId, $signature)
    {
        // In production, verify the signature using payment gateway's method
        // For Razorpay:
        /*
        $attributes = [
            'razorpay_order_id' => $payment->gateway_order_id,
            'razorpay_payment_id' => $gatewayPaymentId,
            'razorpay_signature' => $signature
        ];
        
        $api = new \Razorpay\Api\Api($keyId, $keySecret);
        $api->utility->verifyPaymentSignature($attributes);
        */
        
        return true; // Mock verification
    }

    /**
     * Show payment history
     */
    public function paymentHistory(Request $request)
    {
        $user = Auth::user();
        
        $paymentsQuery = Payment::where('user_id', $user->id)->with('payable');
        
        if ($request->filled('status')) {
            $paymentsQuery->where('status', $request->status);
        }
        
        if ($request->filled('type')) {
            $paymentsQuery->where('payment_type', $request->type);
        }

        $payments = $paymentsQuery->orderBy('created_at', 'desc')->paginate(15);

        return view('payment.history', compact('payments'));
    }

    /**
     * Request refund
     */
    public function requestRefund(Request $request, Payment $payment)
    {
        $user = Auth::user();
        
        if ($user->id !== $payment->user_id) {
            abort(403, 'Unauthorized access.');
        }

        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        if ($payment->status !== 'completed') {
            return back()->with('error', 'Cannot request refund for incomplete payment.');
        }

        // Create refund request (in a real application, this might go to admin approval)
        try {
            // For consultation payments, check refund policy
            if ($payment->payable instanceof Consultation) {
                $consultation = $payment->payable;
                $hoursUntilConsultation = now()->diffInHours($consultation->scheduled_at);
                
                if ($hoursUntilConsultation < 2) {
                    return back()->with('error', 'Refund not allowed within 2 hours of consultation.');
                }
            }

            // In production, this would create a refund request for admin approval
            // For now, we'll process it immediately for demo purposes
            $refundAmount = $payment->total_amount;
            
            $this->paymentService->processRefund($payment, $refundAmount, $request->reason);

            return back()->with('success', 'Refund processed successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Refund request failed: ' . $e->getMessage());
        }
    }
}
