<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\Review;
use App\Services\ConsultationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ConsultationController extends Controller
{
    protected $consultationService;

    public function __construct(ConsultationService $consultationService)
    {
        $this->middleware('auth');
        $this->consultationService = $consultationService;
    }

    /**
     * Show consultation details
     */
    public function show(Consultation $consultation)
    {
        $user = Auth::user();
        
        // Check if user has access to this consultation
        if (!$this->userCanAccessConsultation($user, $consultation)) {
            abort(403, 'Unauthorized access to consultation.');
        }

        $consultation->load(['client.user', 'professional.user', 'reviews']);

        return view('consultation.show', compact('consultation'));
    }

    /**
     * Start a consultation
     */
    public function start(Consultation $consultation)
    {
        $user = Auth::user();
        
        if (!$this->userCanAccessConsultation($user, $consultation)) {
            abort(403, 'Unauthorized access to consultation.');
        }

        if ($consultation->status !== 'scheduled') {
            return back()->with('error', 'Consultation cannot be started in its current status.');
        }

        $this->consultationService->startConsultation($consultation);

        return redirect()->route('consultation.meeting', $consultation)
                        ->with('success', 'Consultation started successfully!');
    }

    /**
     * End a consultation
     */
    public function end(Request $request, Consultation $consultation)
    {
        $user = Auth::user();
        
        if (!$this->userCanAccessConsultation($user, $consultation)) {
            abort(403, 'Unauthorized access to consultation.');
        }

        if ($consultation->status !== 'in_progress') {
            return back()->with('error', 'Consultation is not in progress.');
        }

        $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        $this->consultationService->endConsultation($consultation, $request->notes);

        return back()->with('success', 'Consultation completed successfully!');
    }

    /**
     * Submit review for consultation
     */
    public function review(Request $request, Consultation $consultation)
    {
        $user = Auth::user();
        
        if (!$this->userCanAccessConsultation($user, $consultation)) {
            abort(403, 'Unauthorized access to consultation.');
        }

        if ($consultation->status !== 'completed') {
            return back()->with('error', 'You can only review completed consultations.');
        }

        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:500',
            'criteria_ratings' => 'nullable|array',
        ]);

        // Determine review direction
        if ($user->id === $consultation->client->user_id) {
            // Client reviewing professional
            $fromUserId = $user->id;
            $toUserId = $consultation->professional->user_id;
            $roleFrom = 'client';
        } elseif ($user->id === $consultation->professional->user_id) {
            // Professional reviewing client
            $fromUserId = $user->id;
            $toUserId = $consultation->client->user_id;
            $roleFrom = 'professional';
        } else {
            abort(403, 'Unauthorized to review this consultation.');
        }

        // Check if review already exists
        $existingReview = Review::where('consultation_id', $consultation->id)
                               ->where('from_user_id', $fromUserId)
                               ->first();

        if ($existingReview) {
            return back()->with('error', 'You have already reviewed this consultation.');
        }

        Review::create([
            'consultation_id' => $consultation->id,
            'from_user_id' => $fromUserId,
            'to_user_id' => $toUserId,
            'role_from' => $roleFrom,
            'rating' => $request->rating,
            'comment' => $request->comment,
            'criteria_ratings' => $request->criteria_ratings,
            'approved_by_admin' => $roleFrom === 'client', // Auto-approve client reviews
        ]);

        return back()->with('success', 'Review submitted successfully!');
    }

    /**
     * Join meeting (Jitsi Meet integration)
     */
    public function joinMeeting(Consultation $consultation)
    {
        $user = Auth::user();
        
        if (!$this->userCanAccessConsultation($user, $consultation)) {
            abort(403, 'Unauthorized access to consultation.');
        }

        if (!in_array($consultation->status, ['scheduled', 'in_progress'])) {
            return back()->with('error', 'Meeting is not available for this consultation.');
        }

        // Auto-start consultation if it's scheduled and time has come
        if ($consultation->status === 'scheduled' && now() >= $consultation->scheduled_at) {
            $this->consultationService->startConsultation($consultation);
            $consultation->refresh();
        }

        return view('consultation.meeting', compact('consultation'));
    }

    /**
     * Check if user can access consultation
     */
    private function userCanAccessConsultation($user, $consultation)
    {
        return $user->id === $consultation->client->user_id ||
               $user->id === $consultation->professional->user_id ||
               $user->user_type === 'admin';
    }
}
