<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Review;
use App\Models\Consultation;
use App\Models\Professional;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ReviewController extends Controller
{
    /**
     * Store a new review
     */
    public function store(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user is the client for this consultation
            if ($user->role !== 'client' || $consultation->client_id !== $user->client->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // Check if consultation is completed
            if ($consultation->status !== 'completed') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Can only review completed consultations'
                ], 400);
            }

            // Check if review already exists
            $existingReview = Review::where('consultation_id', $consultation->id)
                ->where('reviewer_id', $user->id)
                ->first();

            if ($existingReview) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Review already submitted for this consultation'
                ], 400);
            }

            $validator = Validator::make($request->all(), [
                'rating' => 'required|integer|min:1|max:5',
                'comment' => 'required|string|min:10|max:1000',
                'would_recommend' => 'required|boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $review = Review::create([
                'consultation_id' => $consultation->id,
                'professional_id' => $consultation->professional_id,
                'reviewer_id' => $user->id,
                'reviewer_type' => 'client',
                'rating' => $request->rating,
                'comment' => $request->comment,
                'would_recommend' => $request->would_recommend,
                'status' => 'pending', // Reviews need approval
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Review submitted successfully',
                'data' => [
                    'review' => $review->load(['professional.user', 'reviewer'])
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to submit review',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user's reviews
     */
    public function myReviews(Request $request)
    {
        try {
            $user = $request->user();

            $query = $user->writtenReviews()->with(['professional.user', 'consultation']);

            // Filter by status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Filter by rating
            if ($request->filled('rating')) {
                $query->where('rating', $request->rating);
            }

            $reviews = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 15));

            return response()->json([
                'status' => 'success',
                'data' => $reviews
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get reviews',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get reviews received by professional
     */
    public function receivedReviews(Request $request)
    {
        try {
            $user = $request->user();

            if ($user->role !== 'professional') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Only professionals can access received reviews'
                ], 403);
            }

            $professional = $user->professional;
            $query = $professional->reviews()->with(['reviewer', 'consultation']);

            // Filter by status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Filter by rating
            if ($request->filled('rating')) {
                $query->where('rating', $request->rating);
            }

            $reviews = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 15));

            // Calculate statistics
            $stats = [
                'total_reviews' => $professional->reviews()->where('status', 'approved')->count(),
                'average_rating' => $professional->reviews()->where('status', 'approved')->avg('rating') ?? 0,
                'rating_distribution' => [
                    '5_star' => $professional->reviews()->where('status', 'approved')->where('rating', 5)->count(),
                    '4_star' => $professional->reviews()->where('status', 'approved')->where('rating', 4)->count(),
                    '3_star' => $professional->reviews()->where('status', 'approved')->where('rating', 3)->count(),
                    '2_star' => $professional->reviews()->where('status', 'approved')->where('rating', 2)->count(),
                    '1_star' => $professional->reviews()->where('status', 'approved')->where('rating', 1)->count(),
                ],
                'recommendation_percentage' => $professional->reviews()
                    ->where('status', 'approved')
                    ->where('would_recommend', true)
                    ->count() / max($professional->reviews()->where('status', 'approved')->count(), 1) * 100
            ];

            return response()->json([
                'status' => 'success',
                'data' => [
                    'reviews' => $reviews,
                    'stats' => $stats
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get received reviews',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get professional reviews (public)
     */
    public function professionalReviews(Request $request, Professional $professional)
    {
        try {
            $query = $professional->reviews()
                ->where('status', 'approved')
                ->with(['reviewer']);

            // Filter by rating
            if ($request->filled('rating')) {
                $query->where('rating', $request->rating);
            }

            $reviews = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 15));

            // Calculate statistics
            $stats = [
                'total_reviews' => $professional->reviews()->where('status', 'approved')->count(),
                'average_rating' => $professional->reviews()->where('status', 'approved')->avg('rating') ?? 0,
                'rating_distribution' => [
                    '5_star' => $professional->reviews()->where('status', 'approved')->where('rating', 5)->count(),
                    '4_star' => $professional->reviews()->where('status', 'approved')->where('rating', 4)->count(),
                    '3_star' => $professional->reviews()->where('status', 'approved')->where('rating', 3)->count(),
                    '2_star' => $professional->reviews()->where('status', 'approved')->where('rating', 2)->count(),
                    '1_star' => $professional->reviews()->where('status', 'approved')->where('rating', 1)->count(),
                ],
                'recommendation_percentage' => $professional->reviews()
                    ->where('status', 'approved')
                    ->where('would_recommend', true)
                    ->count() / max($professional->reviews()->where('status', 'approved')->count(), 1) * 100
            ];

            return response()->json([
                'status' => 'success',
                'data' => [
                    'professional' => $professional->load('user'),
                    'reviews' => $reviews,
                    'stats' => $stats
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get professional reviews',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update review
     */
    public function update(Request $request, Review $review)
    {
        try {
            $user = $request->user();

            // Check if user owns this review
            if ($review->reviewer_id !== $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // Check if review is still editable
            if ($review->status === 'approved') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Cannot edit approved reviews'
                ], 400);
            }

            $validator = Validator::make($request->all(), [
                'rating' => 'sometimes|integer|min:1|max:5',
                'comment' => 'sometimes|string|min:10|max:1000',
                'would_recommend' => 'sometimes|boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $review->update($request->only(['rating', 'comment', 'would_recommend']));

            return response()->json([
                'status' => 'success',
                'message' => 'Review updated successfully',
                'data' => [
                    'review' => $review->fresh()->load(['professional.user'])
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update review',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Report review
     */
    public function report(Request $request, Review $review)
    {
        try {
            $user = $request->user();

            $validator = Validator::make($request->all(), [
                'reason' => 'required|in:inappropriate,spam,fake,harassment,other',
                'description' => 'required_if:reason,other|string|max:500',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Create report record (you'd need a ReviewReport model)
            // For now, we'll just update the review status
            $review->update([
                'status' => 'reported',
                'report_reason' => $request->reason,
                'report_description' => $request->description,
                'reported_by' => $user->id,
                'reported_at' => now(),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Review reported successfully. Our team will investigate.',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to report review',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}