<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Consultation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ConsultationController extends Controller
{
    /**
     * Show consultation details
     */
    public function show(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user has access to this consultation
            if ($user->role === 'professional' && $consultation->professional_id !== $user->professional->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            if ($user->role === 'client' && $consultation->client_id !== $user->client->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $consultation->load([
                'professional.user',
                'client.user',
                'reviews',
                'payment'
            ]);

            return response()->json([
                'status' => 'success',
                'data' => [
                    'consultation' => $consultation
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get consultation details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Start consultation
     */
    public function start(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user is the professional for this consultation
            if ($user->role !== 'professional' || $consultation->professional_id !== $user->professional->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Only the assigned professional can start the consultation'
                ], 403);
            }

            // Check if consultation is scheduled and payment is completed
            if ($consultation->status !== 'scheduled') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Consultation cannot be started. Current status: ' . $consultation->status
                ], 400);
            }

            if ($consultation->payment_status !== 'completed') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Payment must be completed before starting consultation'
                ], 400);
            }

            // Update consultation status
            $consultation->update([
                'status' => 'in_progress',
                'started_at' => now(),
            ]);

            // Generate meeting link (this would integrate with Jitsi Meet or similar)
            $meetingId = 'medex-' . $consultation->id . '-' . time();
            $meetingUrl = "https://meet.jit.si/{$meetingId}";

            $consultation->update([
                'meeting_link' => $meetingUrl,
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Consultation started successfully',
                'data' => [
                    'consultation' => $consultation->fresh(),
                    'meeting_url' => $meetingUrl
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to start consultation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * End consultation
     */
    public function end(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user is the professional for this consultation
            if ($user->role !== 'professional' || $consultation->professional_id !== $user->professional->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Only the assigned professional can end the consultation'
                ], 403);
            }

            // Check if consultation is in progress
            if ($consultation->status !== 'in_progress') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Consultation is not in progress'
                ], 400);
            }

            // Calculate duration
            $startedAt = $consultation->started_at;
            $endedAt = now();
            $duration = $startedAt->diffInMinutes($endedAt);

            // Update consultation
            $consultation->update([
                'status' => 'completed',
                'ended_at' => $endedAt,
                'duration' => $duration,
                'notes' => $request->input('notes'),
                'prescription' => $request->input('prescription'),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Consultation ended successfully',
                'data' => [
                    'consultation' => $consultation->fresh(),
                    'duration_minutes' => $duration
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to end consultation',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Join meeting
     */
    public function joinMeeting(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user has access to this consultation
            $hasAccess = ($user->role === 'professional' && $consultation->professional_id === $user->professional->id) ||
                        ($user->role === 'client' && $consultation->client_id === $user->client->id) ||
                        ($user->role === 'admin');

            if (!$hasAccess) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // Check if consultation is active
            if (!in_array($consultation->status, ['scheduled', 'in_progress'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Consultation is not active'
                ], 400);
            }

            // Check if payment is completed
            if ($consultation->payment_status !== 'completed') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Payment must be completed before joining consultation'
                ], 400);
            }

            // Generate or get existing meeting link
            $meetingUrl = $consultation->meeting_link;
            if (!$meetingUrl) {
                $meetingId = 'medex-' . $consultation->id . '-' . time();
                $meetingUrl = "https://meet.jit.si/{$meetingId}";
                
                $consultation->update([
                    'meeting_link' => $meetingUrl,
                ]);
            }

            return response()->json([
                'status' => 'success',
                'data' => [
                    'meeting_url' => $meetingUrl,
                    'consultation' => $consultation
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get meeting link',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}