<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\Professional;
use App\Models\Consultation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ClientController extends Controller
{
    /**
     * Get client dashboard data
     */
    public function dashboard(Request $request)
    {
        try {
            $user = $request->user();
            $client = $user->client;

            if (!$client) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client profile not found'
                ], 404);
            }

            // Statistics
            $stats = [
                'total_consultations' => $client->consultations()->count(),
                'upcoming_consultations' => $client->consultations()
                    ->where('scheduled_at', '>', now())
                    ->where('status', 'confirmed')
                    ->count(),
                'completed_consultations' => $client->consultations()
                    ->where('status', 'completed')
                    ->count(),
                'total_spent' => $client->consultations()
                    ->where('payment_status', 'completed')
                    ->sum('total_amount'),
            ];

            // Recent consultations
            $recentConsultations = $client->consultations()
                ->with(['professional.user', 'service'])
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();

            // Health tips
            $healthTips = [
                "Regular health checkups can prevent major medical expenses.",
                "Maintain a healthy lifestyle to avoid chronic diseases.",
                "Keep your medical records organized for legal consultations.",
                "Consider health insurance to protect against financial burden.",
                "Early consultation can save both time and money."
            ];

            return response()->json([
                'status' => 'success',
                'data' => [
                    'stats' => $stats,
                    'recent_consultations' => $recentConsultations,
                    'health_tip' => $healthTips[array_rand($healthTips)]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get dashboard data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get client profile
     */
    public function profile(Request $request)
    {
        try {
            $user = $request->user();
            $client = $user->client;

            if (!$client) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client profile not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => [
                    'user' => $user,
                    'client' => $client
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update client profile
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = $request->user();
            $client = $user->client;

            if (!$client) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client profile not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'clinic_hospital_name' => 'sometimes|string|max:255',
                'medical_license' => 'sometimes|string|max:255',
                'address' => 'sometimes|string|max:500',
                'bio' => 'sometimes|string|max:1000',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $client->update($request->only([
                'clinic_hospital_name',
                'medical_license',
                'address',
                'bio'
            ]));

            return response()->json([
                'status' => 'success',
                'message' => 'Profile updated successfully',
                'data' => [
                    'client' => $client->fresh()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search professionals
     */
    public function searchProfessionals(Request $request)
    {
        try {
            $query = Professional::with(['user', 'activeSubscription.subscriptionPlan'])
                ->where('verification_status', 'verified');

            // Filter by profession type
            if ($request->filled('profession_type')) {
                $query->where('profession_type', $request->profession_type);
            }

            // Filter by specializations
            if ($request->filled('specializations')) {
                $specializations = is_array($request->specializations) ? 
                    $request->specializations : [$request->specializations];
                    
                $query->where(function($q) use ($specializations) {
                    foreach ($specializations as $specialization) {
                        $q->orWhereJsonContains('specializations', $specialization);
                    }
                });
            }

            // Filter by experience
            if ($request->filled('min_experience')) {
                $query->where('experience_years', '>=', $request->min_experience);
            }

            // Filter by fee range
            if ($request->filled('max_fee')) {
                $query->where('consultation_fee', '<=', $request->max_fee);
            }

            // Search by name
            if ($request->filled('search')) {
                $searchTerm = $request->search;
                $query->whereHas('user', function($q) use ($searchTerm) {
                    $q->where('name', 'like', "%{$searchTerm}%");
                });
            }

            // Sort options
            switch ($request->get('sort_by', 'rating')) {
                case 'fee_low':
                    $query->orderBy('consultation_fee', 'asc');
                    break;
                case 'fee_high':
                    $query->orderBy('consultation_fee', 'desc');
                    break;
                case 'experience':
                    $query->orderBy('experience_years', 'desc');
                    break;
                case 'rating':
                default:
                    $query->withAvg('reviews', 'rating')
                          ->orderBy('reviews_avg_rating', 'desc');
                    break;
            }

            $professionals = $query->paginate($request->get('per_page', 15));

            return response()->json([
                'status' => 'success',
                'data' => $professionals
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to search professionals',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get consultations
     */
    public function consultations(Request $request)
    {
        try {
            $user = $request->user();
            $client = $user->client;

            if (!$client) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client profile not found'
                ], 404);
            }

            $query = $client->consultations()->with(['professional.user']);

            // Filter by status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Filter by date range
            if ($request->filled('from_date')) {
                $query->whereDate('scheduled_at', '>=', $request->from_date);
            }

            if ($request->filled('to_date')) {
                $query->whereDate('scheduled_at', '<=', $request->to_date);
            }

            $consultations = $query->orderBy('scheduled_at', 'desc')
                ->paginate($request->get('per_page', 15));

            return response()->json([
                'status' => 'success',
                'data' => $consultations
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get consultations',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Book consultation
     */
    public function storeBooking(Request $request, Professional $professional)
    {
        try {
            $user = $request->user();
            $client = $user->client;

            if (!$client) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Client profile not found'
                ], 404);
            }

            if ($professional->verification_status !== 'verified') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional is not verified'
                ], 400);
            }

            $validator = Validator::make($request->all(), [
                'scheduled_at' => 'required|date|after:now',
                'consultation_type' => 'required|in:video,phone,in_person',
                'notes' => 'nullable|string|max:1000',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if professional is available at the requested time
            $conflictingConsultation = $professional->consultations()
                ->where('status', 'scheduled')
                ->whereBetween('scheduled_at', [
                    $request->scheduled_at,
                    date('Y-m-d H:i:s', strtotime($request->scheduled_at . ' +1 hour'))
                ])
                ->exists();

            if ($conflictingConsultation) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional is not available at the requested time'
                ], 409);
            }

            // Calculate fees
            $consultationFee = $professional->consultation_fee;
            $platformFee = $consultationFee * 0.10; // 10% platform fee
            $gst = ($consultationFee + $platformFee) * 0.18; // 18% GST
            $totalAmount = $consultationFee + $platformFee + $gst;

            $consultation = Consultation::create([
                'client_id' => $client->id,
                'professional_id' => $professional->id,
                'scheduled_at' => $request->scheduled_at,
                'consultation_type' => $request->consultation_type,
                'status' => 'scheduled',
                'consultation_fee' => $consultationFee,
                'platform_fee' => $platformFee,
                'gst_amount' => $gst,
                'total_amount' => $totalAmount,
                'professional_fee' => $consultationFee * 0.90, // 90% to professional
                'notes' => $request->notes,
                'payment_status' => 'pending',
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Consultation booked successfully',
                'data' => [
                    'consultation' => $consultation->load(['professional.user']),
                    'next_step' => 'payment_required'
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to book consultation',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}