<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Professional;
use App\Models\Client;
use App\Models\Consultation;
use App\Models\Payment;
use App\Models\Review;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (Auth::user()->user_type !== 'admin') {
                abort(403, 'Admin access required.');
            }
            return $next($request);
        });
    }

    /**
     * Admin Dashboard
     */
    public function dashboard()
    {
        // Gather comprehensive statistics
        $stats = [
            'total_users' => User::count(),
            'new_users_this_month' => User::where('created_at', '>=', now()->startOfMonth())->count(),
            'active_professionals' => Professional::whereHas('user', function($q) {
                $q->where('status', 'active');
            })->count(),
            'verified_professionals' => Professional::where('verification_status', 'verified')->count(),
            'total_consultations' => Consultation::count(),
            'consultations_this_month' => Consultation::where('created_at', '>=', now()->startOfMonth())->count(),
            'total_revenue' => Payment::where('status', 'completed')->sum('total_amount'),
            'revenue_this_month' => Payment::where('status', 'completed')
                                          ->where('created_at', '>=', now()->startOfMonth())
                                          ->sum('total_amount'),
            'active_subscriptions' => Subscription::where('status', 'active')->count(),
            'average_rating' => Review::where('status', 'approved')->avg('rating') ?? 0,
            'pending_approvals' => Professional::where('verification_status', '!=', 'verified')->count(),
            'reported_reviews' => DB::table('review_reports')->where('status', 'pending')->count(),
        ];

        // Chart data for last 30 days
        $chartData = $this->getChartData();

        // Recent activities
        $recentActivities = $this->getRecentActivities();

        // Top professionals by earnings
        $topProfessionals = $this->getTopProfessionals();

        return view('admin.dashboard', compact('stats', 'chartData', 'recentActivities', 'topProfessionals'));
    }

    /**
     * Manage Professionals
     */
    public function professionals(Request $request)
    {
        $query = Professional::with(['user', 'activeSubscription']);

        // Filter by verification status
        if ($request->filled('status')) {
            $query->where('verification_status', $request->status);
        }

        // Filter by specialty
        if ($request->filled('specialty')) {
            $query->where('specialty', $request->specialty);
        }

        // Search by name
        if ($request->filled('search')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%');
            });
        }

        $professionals = $query->orderBy('created_at', 'desc')->paginate(15);

        $specialties = Professional::distinct()->pluck('specializations')->filter()->flatten()->unique();

        return view('admin.professionals', compact('professionals', 'specialties'));
    }

    /**
     * Verify Professional
     */
    public function verifyProfessional(Professional $professional)
    {
        $professional->update([
            'verification_status' => 'verified',
            'verified_at' => now()
        ]);

        return redirect()->back()->with('success', 'Professional verified successfully.');
    }

    /**
     * Manage Clients
     */
    public function clients(Request $request)
    {
        $query = Client::with('user');

        if ($request->filled('search')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%');
            });
        }

        $clients = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.clients', compact('clients'));
    }

    /**
     * Manage Consultations
     */
    public function consultations(Request $request)
    {
        $query = Consultation::with(['professional.user', 'client.user', 'payment']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('scheduled_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('scheduled_at', '<=', $request->date_to);
        }

        $consultations = $query->orderBy('scheduled_at', 'desc')->paginate(15);

        return view('admin.consultations', compact('consultations'));
    }

    /**
     * Manage Payments
     */
    public function payments(Request $request)
    {
        $query = Payment::with(['user', 'payable']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('payment_type', $request->type);
        }

        $payments = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.payments', compact('payments'));
    }

    /**
     * Get chart data for dashboard
     */
    private function getChartData()
    {
        $days = 30;
        $dates = collect(range(0, $days-1))->map(function ($i) {
            return now()->subDays($i)->format('M d');
        })->reverse()->values();

        // Revenue data
        $revenueData = collect(range(0, $days-1))->map(function ($i) {
            $date = now()->subDays($i)->toDateString();
            return Payment::where('status', 'completed')
                         ->whereDate('created_at', $date)
                         ->sum('total_amount');
        })->reverse()->values();

        // Consultation data
        $consultationData = collect(range(0, $days-1))->map(function ($i) {
            $date = now()->subDays($i)->toDateString();
            return Consultation::whereDate('created_at', $date)->count();
        })->reverse()->values();

        return [
            'revenue' => [
                'labels' => $dates,
                'data' => $revenueData
            ],
            'consultations' => [
                'labels' => $dates,
                'data' => $consultationData
            ]
        ];
    }

    /**
     * Get recent activities
     */
    private function getRecentActivities()
    {
        $activities = [];

        // Recent user registrations
        $recentUsers = User::latest()->take(3)->get();
        foreach ($recentUsers as $user) {
            $activities[] = [
                'icon' => 'user-plus',
                'color' => 'success',
                'description' => "New {$user->user_type} registered: {$user->name}",
                'time' => $user->created_at->diffForHumans()
            ];
        }

        // Recent consultations
        $recentConsultations = Consultation::with(['professional.user', 'client.user'])
                                         ->latest()->take(3)->get();
        foreach ($recentConsultations as $consultation) {
            $activities[] = [
                'icon' => 'calendar-check',
                'color' => 'info',
                'description' => "Consultation booked with {$consultation->professional->user->name}",
                'time' => $consultation->created_at->diffForHumans()
            ];
        }

        // Recent payments
        $recentPayments = Payment::with('user')->where('status', 'completed')
                               ->latest()->take(2)->get();
        foreach ($recentPayments as $payment) {
            $activities[] = [
                'icon' => 'credit-card',
                'color' => 'success',
                'description' => "Payment of ₹{$payment->total_amount} by {$payment->user->name}",
                'time' => $payment->created_at->diffForHumans()
            ];
        }

        // Sort by time and limit to 10
        return collect($activities)->sortByDesc('time')->take(10)->values()->toArray();
    }

    /**
     * Get top professionals by earnings
     */
    private function getTopProfessionals()
    {
        return DB::table('professionals')
            ->join('users', 'professionals.user_id', '=', 'users.id')
            ->leftJoin('consultations', 'professionals.id', '=', 'consultations.professional_id')
            ->leftJoin('payments', function($join) {
                $join->on('consultations.id', '=', 'payments.payable_id')
                     ->where('payments.payable_type', '=', 'App\\Models\\Consultation')
                     ->where('payments.status', '=', 'completed');
            })
            ->select(
                'professionals.id',
                'users.name',
                'professionals.profession_type as specialty',
                'professionals.rating',
                DB::raw('COALESCE(SUM(payments.total_amount), 0) as total_earnings')
            )
            ->groupBy('professionals.id', 'users.name', 'professionals.profession_type', 'professionals.rating')
            ->orderByDesc('total_earnings')
            ->take(5)
            ->get()
            ->map(function ($professional) {
                return [
                    'name' => $professional->name,
                    'specialty' => $professional->specialty,
                    'earnings' => $professional->total_earnings ?? 0,
                    'rating' => number_format($professional->rating ?? 0, 1)
                ];
            })
            ->toArray();
    }

    /**
     * Manage Reviews
     */
    public function reviews(Request $request)
    {
        $query = Review::with(['user', 'professional.user'])->orderBy('created_at', 'desc');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search functionality
        if ($request->filled('search')) {
            $query->where(function($q) use ($request) {
                $q->where('comment', 'like', '%' . $request->search . '%')
                  ->orWhereHas('user', function($userQuery) use ($request) {
                      $userQuery->where('name', 'like', '%' . $request->search . '%');
                  });
            });
        }

        $reviews = $query->paginate(15);
        
        return view('admin.reviews', compact('reviews'));
    }

    /**
     * Approve Review
     */
    public function approveReview(Review $review)
    {
        $review->update(['status' => 'approved']);
        
        return redirect()->back()->with('success', 'Review approved successfully!');
    }

    /**
     * Manage Subscription Plans
     */
    public function subscriptionPlans(Request $request)
    {
        $subscriptionPlans = SubscriptionPlan::orderBy('created_at', 'desc')->get();
        
        return view('admin.subscription-plans', compact('subscriptionPlans'));
    }

    /**
     * Store new subscription plan
     */
    public function storeSubscriptionPlan(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'monthly_price' => 'required|numeric|min:0',
            'yearly_price' => 'nullable|numeric|min:0',
            'commission_percentage' => 'nullable|numeric|min:0|max:100',
            'max_consultations_per_month' => 'nullable|integer|min:0',
            'features' => 'nullable|string',
            'is_active' => 'required|boolean',
        ]);

        $features = null;
        if ($request->features) {
            $features = array_filter(array_map('trim', explode("\n", $request->features)));
        }

        SubscriptionPlan::create([
            'name' => $request->name,
            'slug' => \Str::slug($request->name),
            'description' => $request->description,
            'monthly_price' => $request->monthly_price,
            'yearly_price' => $request->yearly_price,
            'commission_percentage' => $request->commission_percentage ?? 10,
            'max_consultations_per_month' => $request->max_consultations_per_month,
            'features' => $features,
            'priority_listing' => $request->has('priority_listing'),
            'featured_profile' => $request->has('featured_profile'),
            'dedicated_support' => $request->has('dedicated_support'),
            'is_active' => $request->is_active,
            'sort_order' => SubscriptionPlan::max('sort_order') + 1,
        ]);

        return redirect()->back()->with('success', 'Subscription plan created successfully!');
    }

    /**
     * Update subscription plan
     */
    public function updateSubscriptionPlan(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'monthly_price' => 'required|numeric|min:0',
            'yearly_price' => 'nullable|numeric|min:0',
            'commission_percentage' => 'nullable|numeric|min:0|max:100',
            'max_consultations_per_month' => 'nullable|integer|min:0',
            'features' => 'nullable|string',
            'is_active' => 'required|boolean',
        ]);

        $features = null;
        if ($request->features) {
            $features = array_filter(array_map('trim', explode("\n", $request->features)));
        }

        $subscriptionPlan->update([
            'name' => $request->name,
            'slug' => \Str::slug($request->name),
            'description' => $request->description,
            'monthly_price' => $request->monthly_price,
            'yearly_price' => $request->yearly_price,
            'commission_percentage' => $request->commission_percentage ?? 10,
            'max_consultations_per_month' => $request->max_consultations_per_month,
            'features' => $features,
            'priority_listing' => $request->has('priority_listing'),
            'featured_profile' => $request->has('featured_profile'),
            'dedicated_support' => $request->has('dedicated_support'),
            'is_active' => $request->is_active,
        ]);

        return redirect()->back()->with('success', 'Subscription plan updated successfully!');
    }

    /**
     * Delete subscription plan
     */
    public function destroySubscriptionPlan(SubscriptionPlan $subscriptionPlan)
    {
        // Check if any active subscriptions exist for this plan
        $activeSubscriptions = Subscription::where('subscription_plan_id', $subscriptionPlan->id)
                                         ->where('status', 'active')
                                         ->count();
        
        if ($activeSubscriptions > 0) {
            return redirect()->back()->with('error', 'Cannot delete plan with active subscriptions!');
        }

        $subscriptionPlan->delete();
        
        return redirect()->back()->with('success', 'Subscription plan deleted successfully!');
    }
}
